
# pylint: disable=no-name-in-module
# pylint: disable=broad-exception-caught
"""
NOTICE OF LICENSE.

Copyright 2025 @AnabolicsAnonymous

Licensed under the Affero General Public License v3.0 (AGPL-3.0)
"""
import json
import time
import os
import tempfile
from decimal import Decimal
from collections import Counter
from scapy.all import rdpcap, IP, IPv6, TCP, UDP

def convert_decimal(obj):
    """Recursively convert elements to JSON-safe primitives."""
    if isinstance(obj, Decimal):
        return float(obj)
    if isinstance(obj, set):
        return list(obj)
    if isinstance(obj, dict):
        return {key: convert_decimal(value) for key, value in obj.items()}
    if isinstance(obj, list):
        return [convert_decimal(item) for item in obj]
    return obj

def update_ip_info(pcap_info, packet, ip_src_key, ip_src, counters):
    """Update packet information in the pcap_info dict for a given IP source."""
    if ip_src not in pcap_info[ip_src_key]:
        pcap_info[ip_src_key][ip_src] = {
            "packets_from_ip": 0,
            "pps_from_ip": 0,
            "packet_types_from_ip": Counter(),
            "syn": 0, "ack": 0, "synack": 0, "fin": 0, "rst": 0, "psh": 0, "urg": 0, "ece": 0, "cwr": 0,
            "first_timestamp": round(packet.time, 2),
            "last_timestamp": round(packet.time, 2),
            "source_ports": set(),
            "destination_ports": set(),
            "destination_ips": set()
        }

    ip_info = pcap_info[ip_src_key][ip_src]
    ip_info["packets_from_ip"] += 1
    ip_info["last_timestamp"] = round(packet.time, 2)

    packet_type = packet.__class__.__name__
    ip_info["packet_types_from_ip"][packet_type] += 1

    if TCP in packet or UDP in packet:
        protocol = TCP if TCP in packet else UDP
        process_transport_protocol(ip_info, packet, protocol, counters)

    process_ip_or_ipv6(ip_info, packet, counters)
    counters['source_ip'][ip_src] += 1

def process_transport_protocol(ip_info, packet, protocol, counters):
    """Process source and destination ports for TCP or UDP packets."""
    src_port = int(packet[protocol].sport)
    dst_port = int(packet[protocol].dport)
    ip_info["source_ports"].add(src_port)
    ip_info["destination_ports"].add(dst_port)

    counters['source_port'][src_port] += 1
    counters['dest_port'][dst_port] += 1

    if TCP in packet:
        flags = str(packet[TCP].flags)
        flag_map = {
            "S": "syn", "A": "ack", "SA": "synack", "F": "fin", "R": "rst",
            "P": "psh", "U": "urg", "E": "ece", "C": "cwr"
        }
        for flag, flag_name in flag_map.items():
            if flag in flags:
                ip_info[flag_name] += 1

def process_ip_or_ipv6(ip_info, packet, counters):
    """Process destination IP/IPv6 for the given packet."""
    dst_ip = packet[IP].dst if IP in packet else packet[IPv6].dst
    ip_info["destination_ips"].add(dst_ip)
    counters['dest_ip'][dst_ip] += 1

def extract_pcap_details(pcap):
    """Extract details from a PCAP: addresses, ports, stats."""
    packets = rdpcap(pcap)

    pcap_info = {
        "pcap_name": os.path.basename(pcap),
        "pcap_size_bytes": os.path.getsize(pcap),
        "pcap_packets_captured": len(packets),
        "pcap_date_captured": time.ctime(),
        "pcap_duration_seconds": round((packets[-1].time - packets[0].time), 2) if packets else 0,
        "most_common_source_ip": None,
        "most_common_dest_ip": None,
        "most_common_port_from": None,
        "most_common_port_to": None,
        "ipv4_addresses": {},
        "ipv6_addresses": {}
    }

    counters = {'source_ip': Counter(), 'dest_ip': Counter(), 'source_port': Counter(), 'dest_port': Counter()}

    for packet in packets:
        if IP in packet:
            update_ip_info(pcap_info, packet, 'ipv4_addresses', packet[IP].src, counters)
        elif IPv6 in packet:
            update_ip_info(pcap_info, packet, 'ipv6_addresses', packet[IPv6].src, counters)

    pcap_info["most_common_source_ip"] = counters['source_ip'].most_common(1)[0][0] if counters['source_ip'] else None
    pcap_info["most_common_dest_ip"] = counters['dest_ip'].most_common(1)[0][0] if counters['dest_ip'] else None
    pcap_info["most_common_port_from"] = int(counters['source_port'].most_common(1)[0][0]) if counters['source_port'] else None
    pcap_info["most_common_port_to"] = int(counters['dest_port'].most_common(1)[0][0]) if counters['dest_port'] else None

    for ip_address in pcap_info['ipv4_addresses'].values():
        if ip_address["first_timestamp"] != ip_address["last_timestamp"]:
            ip_address["pps_from_ip"] = round(ip_address["packets_from_ip"] /
                                              max(0.001, (ip_address["last_timestamp"] - ip_address["first_timestamp"])), 2)

    for ipv6_address in pcap_info['ipv6_addresses'].values():
        if ipv6_address["first_timestamp"] != ipv6_address["last_timestamp"]:
            ipv6_address["pps_from_ip"] = round(ipv6_address["packets_from_ip"] /
                                                max(0.001, (ipv6_address["last_timestamp"] - ipv6_address["first_timestamp"])), 2)

    return pcap_info

def save_pcap_info_to_json(pcap_info, export_json):
    """Write the processed PCAP info to JSON atomically."""
    pcap_info = convert_decimal(pcap_info)
    os.makedirs(os.path.dirname(export_json), exist_ok=True)
    with tempfile.NamedTemporaryFile("w", delete=False, encoding="utf-8", dir=os.path.dirname(export_json)) as tf:
        json.dump(pcap_info, tf, indent=4)
        temp_name = tf.name
    os.replace(temp_name, export_json)

def main(pcap, export_json):
    """Module entry point, returns True if no errors occurred."""
    try:
        pcap_file = extract_pcap_details(pcap)
        save_pcap_info_to_json(pcap_file, export_json)
        return True
    except Exception as e:
        return e
