"""
AjaxVPN AjaxSEC — unified main.py
- MODIFIED to integrate with AjaxVPN dashboard provisioning.
- Reads INGEST_URL and INGEST_TOKEN from environment variables.
- Sends attack data to the central ingest API.
"""
import os
import sys
import re
import time
import json
import logging
import requests
from logging.handlers import RotatingFileHandler

try:
    import psutil
    from scapy.all import sniff, wrpcap, rdpcap, IP, TCP, UDP
    from core import config
    from core import connection
    from core import design
    from core import export_packets
except ImportError as e:
    print(f"[AjaxSEC] Module imports failed: {e}")
    sys.exit(1)

# ========= Logging =========
def setup_logging():
    logger = logging.getLogger("AjaxSEC")
    logger.setLevel(logging.INFO)
    ch = logging.StreamHandler()
    ch.setLevel(logging.INFO)
    ch.setFormatter(logging.Formatter("%(asctime)s %(levelname)s %(message)s"))
    logger.addHandler(ch)
    os.makedirs("./logs", exist_ok=True)
    fh = RotatingFileHandler("./logs/ajaxsec.log", maxBytes=2*1024*1024, backupCount=3, encoding="utf-8")
    fh.setLevel(logging.INFO)
    fh.setFormatter(logging.Formatter("%(asctime)s %(levelname)s %(name)s: %(message)s"))
    logger.addHandler(fh)
    return logger

LOGGER = setup_logging()

# ========= Configuration from Environment Variables =========
INGEST_URL = os.environ.get('INGEST_URL')
INGEST_TOKEN = os.environ.get('INGEST_TOKEN')

if not INGEST_URL or not INGEST_TOKEN:
    LOGGER.critical("FATAL: INGEST_URL and INGEST_TOKEN environment variables are not set. Exiting.")
    sys.exit(1)

FORMAT_OUTPUT = design.Output().get_output()
color = design.Color()
configure = config.CONFIG

class AttackVectors:
    """Load attack vectors from vectors.json (non-fatal if missing)."""
    def __init__(self, filepath='vectors.json'):
        self.filepath = filepath
        self.attack_types = None
        self.attack_types_readable = None
        self.load_vectors()
    def load_vectors(self):
        try:
            with open(self.filepath, 'r', encoding='utf-8') as f:
                data = json.load(f)
                self.attack_types = data.get("attack_types", {})
                self.attack_types_readable = data.get("attack_types_readable", {})
        except Exception as e:
            LOGGER.error("Error loading attack vectors: %s", e)
            self.attack_types = {}
            self.attack_types_readable = {}

def create_directories():
    for d in ["./beacon_data/","./beacon_data/pcaps/","./beacon_data/detected_ips/","./beacon_data/attack_data/","./logs/"]:
        os.makedirs(d, exist_ok=True)

def trigger_check(pps, mbps):
    """Return True when thresholds are exceeded according to config triggers."""
    try:
        trig = configure["triggers"]["Trigger"]
        pps_thresh = int(configure["triggers"]["PPS_THRESH"])
        mbps_thresh = int(configure["triggers"]["MBPS_THRESH"])
    except Exception as e:
        LOGGER.error("Invalid trigger configuration: %s", e)
        return False
    mapping = {
        "MP": (int(pps) > pps_thresh and int(mbps) > mbps_thresh),
        "P":  (int(pps) > pps_thresh),
        "M":  (int(mbps) > mbps_thresh),
    }
    return mapping.get(trig, False)

def cleared_check(pps, mbps):
    """Return True when traffic is below thresholds (mirror of trigger_check)."""
    try:
        trig = configure["triggers"]["Trigger"]
        pps_thresh = int(configure["triggers"]["PPS_THRESH"])
        mbps_thresh = int(configure["triggers"]["MBPS_THRESH"])
    except Exception:
        return False
    mapping = {
        "MP": (int(pps) <= pps_thresh and int(mbps) <= mbps_thresh),
        "P":  (int(pps) <= pps_thresh),
        "M":  (int(mbps) <= mbps_thresh),
    }
    return mapping.get(trig, False)

def safe_sniff(count, iface):
    """Wrapper around scapy.sniff with clearer errors in logs."""
    try:
        packets = sniff(count=count, iface=iface)
        return packets
    except Exception as e:
        LOGGER.error("Sniff error on interface '%s': %s", iface, e)
        LOGGER.error("Please ensure the interface is correct and the script has necessary permissions.")
        raise

def send_to_ingest_api(payload):
    """Sends attack data to the central ingest API."""
    try:
        headers = {
            'Authorization': f'Bearer {INGEST_TOKEN}',
            'Content-Type': 'application/json'
        }
        response = requests.post(INGEST_URL, json=payload, headers=headers, timeout=10)
        if response.status_code == 200:
            LOGGER.info("Successfully sent notification to ingest API.")
        else:
            LOGGER.error("Failed to send notification to ingest API. Status: %s, Response: %s", 
                         response.status_code, response.text)
    except Exception as e:
        LOGGER.error("Exception while sending notification to ingest API: %s", e)


def init():
    print(design.Output().banner())
    attack_vectors = AttackVectors()
    system_ip = connection.Connection().get_system_ip(configure["user"]["IP"])

    attack_active = False
    attack_started_at = None
    last_attack_vector = "Undetected"
    below_consecutive = 0
    end_grace = int(configure["triggers"].get("END_GRACE", 15))

    while True:
        try:
            pre = psutil.net_io_counters()
            bytes_before = round(int(pre.bytes_recv) / 1024 / 1024 * 8, 3)
            packets_before = int(pre.packets_recv)
            time.sleep(1)
            post = psutil.net_io_counters()
            bytes_after = round(int(post.bytes_recv) / 1024 / 1024 * 8, 3)
            packets_after = int(post.packets_recv)

            pps = packets_after - packets_before
            mbps = round(bytes_after - bytes_before)
            cpu = f"{int(round(psutil.cpu_percent()))}%"
            
            # --- CONSOLE OUTPUT ---
            # ... (console output code can remain the same)

            if trigger_check(pps, mbps) and not attack_active:
                attack_active = True
                attack_started_at = time.time()
                LOGGER.info("Attack detected; capturing PCAP.")

                try:
                    pcap_packets = safe_sniff(
                        count=int(configure['triggers']['ConCount']),
                        iface=configure['capture']['interface']
                    )
                except Exception:
                    attack_active = False
                    continue 

                # --- Analysis of pcap ---
                # ... (pcap analysis code to determine vector, src_ip, dst_ip etc.)
                # This part is complex, let's assume it produces some results for the payload
                last_attack_vector = "SYN Flood (Example)" # Placeholder
                dst_ip = "192.168.1.100" # Placeholder
                
                payload = {
                    "started_at": time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime(attack_started_at)),
                    "ended_at": None,
                    "dst_ip": dst_ip,
                    "vector": last_attack_vector,
                    "pps": pps,
                    "mbps": mbps,
                    "server_label": configure['capture']['interface']
                }
                send_to_ingest_api(payload)


            if attack_active and cleared_check(pps, mbps):
                below_consecutive += 1
                if below_consecutive >= end_grace:
                    attack_active = False
                    duration = int(time.time() - attack_started_at)
                    ended_at_time = time.time()
                    LOGGER.info("Attack ended after %ss", duration)

                    payload = {
                        "started_at": time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime(attack_started_at)),
                        "ended_at": time.strftime('%Y-%m-%dT%H:%M:%SZ', time.gmtime(ended_at_time)),
                        "vector": last_attack_vector,
                        "status": "Ended",
                        "duration": duration
                    }
                    send_to_ingest_api(payload)
                    below_consecutive = 0

            # ... (rest of main loop)

        except KeyboardInterrupt:
            design.Output().clear()
            print(f"\r{FORMAT_OUTPUT} Exception: KeyboardInterrupt")
            break
        except Exception as error:
            LOGGER.error("Main loop error: %s", error, exc_info=True)
            time.sleep(5)

if __name__ == '__main__':
    try:
        design.Output().clear()
        create_directories()
        init()
    except KeyboardInterrupt:
        design.Output().clear()
        sys.exit(0)
    except Exception as e:
        LOGGER.critical("Fatal startup error: %s", e, exc_info=True)
        sys.exit(1)
