
# pylint: disable=broad-exception-caught
"""
AjaxSEC main with "attack ended" detection & notification.

- Tracks when an attack starts and ends using the same thresholds you already have.
- Sends a second webhook when traffic has dropped below thresholds for a grace period.
"""
import os
import sys
import re
import time
import json
import logging
from logging.handlers import RotatingFileHandler

try:
    import psutil
    from scapy.all import sniff, wrpcap, rdpcap, IP, TCP, UDP
    from core import config
    from core import connection
    from core import notifications as notis
    from core import design
    from core import export_packets
except ImportError as e:
    print(f"[AjaxSEC] Module imports failed: {e}")
    sys.exit(1)

def setup_logging():
    logger = logging.getLogger("AjaxSEC")
    logger.setLevel(logging.INFO)
    ch = logging.StreamHandler()
    ch.setLevel(logging.INFO)
    ch.setFormatter(logging.Formatter("%(asctime)s %(levelname)s %(message)s"))
    logger.addHandler(ch)
    os.makedirs("./logs", exist_ok=True)
    fh = RotatingFileHandler("./logs/ajaxsec.log", maxBytes=2*1024*1024, backupCount=3, encoding="utf-8")
    fh.setLevel(logging.INFO)
    fh.setFormatter(logging.Formatter("%(asctime)s %(levelname)s %(name)s: %(message)s"))
    logger.addHandler(fh)
    return logger

LOGGER = setup_logging()

FORMAT_OUTPUT = design.Output().get_output()
color = design.Color()
configure = config.CONFIG

class AttackVectors:
    def __init__(self, filepath='vectors.json'):
        self.filepath = filepath
        self.attack_types = None
        self.attack_types_readable = None
        self.load_vectors()
    def load_vectors(self):
        try:
            with open(self.filepath, 'r', encoding='utf-8') as f:
                data = json.load(f)
                self.attack_types = data.get("attack_types", {})
                self.attack_types_readable = data.get("attack_types_readable", {})
        except Exception as e:
            LOGGER.error("Error loading attack vectors: %s", e)
            self.attack_types = {}
            self.attack_types_readable = {}

def create_directories():
    for d in ["./beacon_data/","./beacon_data/pcaps/","./beacon_data/detected_ips/","./beacon_data/attack_data/","./logs/"]:
        os.makedirs(d, exist_ok=True)

def trigger_check(pps, mbps):
    try:
        trig = configure["triggers"]["Trigger"]
        pps_thresh = int(configure["triggers"]["PPS_THRESH"])
        mbps_thresh = int(configure["triggers"]["MBPS_THRESH"])
    except Exception as e:
        LOGGER.error("Invalid trigger configuration: %s", e)
        return False
    mapping = {
        "MP": (int(pps) > pps_thresh and int(mbps) > mbps_thresh),
        "P":  (int(pps) > pps_thresh),
        "M":  (int(mbps) > mbps_thresh),
    }
    return mapping.get(trig, False)

def cleared_check(pps, mbps):
    """Return True when traffic is below the trigger thresholds (mirror of trigger_check)."""
    try:
        trig = configure["triggers"]["Trigger"]
        pps_thresh = int(configure["triggers"]["PPS_THRESH"])
        mbps_thresh = int(configure["triggers"]["MBPS_THRESH"])
    except Exception:
        return False
    mapping = {
        "MP": (int(pps) <= pps_thresh and int(mbps) <= mbps_thresh),
        "P":  (int(pps) <= pps_thresh),
        "M":  (int(mbps) <= mbps_thresh),
    }
    return mapping.get(trig, False)

def safe_sniff(count, iface):
    from scapy.all import sniff
    return sniff(count=count, iface=iface)

def display_export_data(export_json):
    try:
        with open(export_json, 'r', encoding='utf-8') as f:
            export_data = json.load(f)
        if "ipv4_addresses" in export_data or "ipv6_addresses" in export_data:
            ipv4_count = len(export_data.get("ipv4_addresses", {}))
            ipv6_count = len(export_data.get("ipv6_addresses", {}))
            total_ips = ipv4_count + ipv6_count
            print(f"{FORMAT_OUTPUT}{'Total Unique IPs:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{total_ips:^{15}}{color.OK_BLUE}]{color.RESET}")
    except Exception as e:
        LOGGER.warning("Error reading export JSON: %s", e)

def init():
    print(design.Output().banner())
    discord_notifier = notis.DiscordNotifier()
    attack_vectors = AttackVectors()
    system_ip = connection.Connection().get_system_ip(configure["user"]["IP"])

    # --- new state vars for "attack ended" ---
    attack_active = False
    attack_started_at = None
    last_export_json = None
    last_attack_vector = "Undetected"
    below_consecutive = 0
    end_grace = int(configure["triggers"].get("END_GRACE", 15))  # seconds below thresholds to consider ended

    while True:
        try:
            pre = psutil.net_io_counters()
            bytes_before = round(int(pre.bytes_recv) / 1024 / 1024 * 8, 3)
            packets_before = int(pre.packets_recv)
            time.sleep(1)
            post = psutil.net_io_counters()
            bytes_after = round(int(post.bytes_recv) / 1024 / 1024 * 8, 3)
            packets_after = int(post.packets_recv)

            pps = packets_after - packets_before
            mbps = round(bytes_after - bytes_before)
            cpu = f"{int(round(psutil.cpu_percent()))}%"
            conn_status, ping_time = connection.Connection().get_connection_status()

            print(f"{FORMAT_OUTPUT}{'IP Address:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{system_ip:^{15}}{color.OK_BLUE}]{color.RESET}")
            print(f"{FORMAT_OUTPUT}{'CPU:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{cpu:^{15}}{color.OK_BLUE}]{color.RESET}")
            print(f"{FORMAT_OUTPUT}{'Connection:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{conn_status:^{15}}{color.OK_BLUE}]{color.RESET}")
            if ping_time is not None:
                print(f"{FORMAT_OUTPUT}{'Ping:':>23}{color.OK_BLUE} "
                      f"[{color.FAIL}{f'{ping_time}ms':^{15}}{color.OK_BLUE}]{color.RESET}")
            print(f"{FORMAT_OUTPUT}{'Megabits Per Second:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{mbps:^{15}}{color.OK_BLUE}]{color.RESET}")
            print(f"{FORMAT_OUTPUT}{'Packets Per Second:':>23}{color.OK_BLUE} "
                  f"[{color.FAIL}{pps:^{15}}{color.OK_BLUE}]{color.RESET}")

            # ---- start detection ----
            if trigger_check(pps, mbps) and not attack_active:
                attack_active = True
                attack_started_at = time.time()
                print(f"{FORMAT_OUTPUT}{'Traffic Increased:':>23}{color.OK_BLUE} "
                      f"[{color.FAIL}{'Capturing...':^{15}}{color.OK_BLUE}]{color.RESET}")
                LOGGER.info("Attack started; capturing PCAP")

                try:
                    timestamp = design.Output().get_time()
                    safe_timestamp = re.sub(r'[^A-Za-z0-9_.-]', '_', timestamp)
                    pcap_file = f"./beacon_data/pcaps/capture.{safe_timestamp}.pcap"
                    packets = safe_sniff(count=int(configure['triggers']['ConCount']),
                                         iface=configure['capture']['interface'])
                    wrpcap(pcap_file, packets)

                    attack_type_list = f"./beacon_data/attack_data/proto.{safe_timestamp}.txt"
                    packets = rdpcap(pcap_file)
                    with open(attack_type_list, "w", encoding="utf-8") as fh:
                        fh.write("ip.proto\ttcp.flags\tudp.srcport\ttcp.srcport\t")
                        for pkt in packets:
                            ip_proto = tcp_flags = udp_srcport = tcp_srcport = ""
                            if IP in pkt: ip_proto = str(pkt[IP].proto)
                            if TCP in pkt:
                                try: tcp_flags = hex(int(pkt[TCP].flags))
                                except Exception: tcp_flags = ""
                                tcp_srcport = str(pkt[TCP].sport)
                            if UDP in pkt: udp_srcport = str(pkt[UDP].sport)
                            fh.write(f"{ip_proto}\t{tcp_flags}\t{udp_srcport}\t{tcp_srcport}\n")

                    sys.stdout.write('\x1b[1A'); sys.stdout.write('\x1b[2K')
                    print(f"{FORMAT_OUTPUT}{'Traffic Increased:':>23}{color.OK_BLUE} "
                          f"[{color.FAIL}{'Captured!':^{15}}{color.OK_BLUE}]{color.RESET}")
                except Exception as e:
                    LOGGER.error("Capture error: %s", e)
                    attack_active = False
                    continue

                # analyze
                try:
                    with open(attack_type_list, "r", encoding="utf-8") as fh:
                        capture_file = fh.read()
                except Exception:
                    capture_file = ""

                def pct(n, total):
                    try: return round(100.0 * float(n) / float(total), 1)
                    except Exception: return 0.0
                con_count = int(configure["triggers"].get("ConCount", 1))
                min_occ = int(configure["triggers"].get("Attack_occurrences", 1))

                disp = []
                webhook_disp = []
                for key, pattern in (attack_vectors.attack_types or {}).items():
                    number = capture_file.count(pattern)
                    if number > min_occ:
                        disp.append(f"{key} ({pct(number, con_count)}%)")
                for key, pattern in (attack_vectors.attack_types_readable or {}).items():
                    number = capture_file.count(pattern)
                    if number > min_occ:
                        webhook_disp.append(f"{key} ({pct(number, con_count)}%)")

                if not disp: disp = ["Undetected"]
                if not webhook_disp: webhook_disp = ["Undetected"]
                last_attack_vector = ' '.join(webhook_disp)

                # export json
                try:
                    timestamp = design.Output().get_time()
                    safe_timestamp = re.sub(r'[^A-Za-z0-9_.-]', '_', timestamp)
                    export_json = f"./beacon_data/detected_ips/export.{safe_timestamp}.json"
                    ok = export_packets.main(pcap_file, export_json=export_json)
                    if ok is True:
                        display_export_data(export_json)
                        last_export_json = export_json
                except Exception as e:
                    LOGGER.error("Export failure: %s", e)

                attack_data = {"pps": pps, "mbps": mbps, "cpu": cpu, "pcap": pcap_file,
                               "attack_vector": last_attack_vector, "status": "Detected"}
                notis.DiscordNotifier().send_notification(attack_data, last_export_json)

            # ---- end detection ----
            if attack_active:
                if cleared_check(pps, mbps):
                    below_consecutive += 1
                else:
                    below_consecutive = 0

                if below_consecutive >= end_grace:
                    attack_active = False
                    duration = int(time.time() - (attack_started_at or time.time()))
                    LOGGER.info("Attack ended after %ss", duration)
                    # Send end webhook (reuse last data where possible)
                    end_data = {
                        "pps": pps, "mbps": mbps, "cpu": cpu,
                        "pcap": "—",  # no new capture at end
                        "attack_vector": last_attack_vector,
                        "status": "Ended",
                        "duration": duration
                    }
                    notis.DiscordNotifier().send_notification(end_data, last_export_json)
                    print(f"{FORMAT_OUTPUT}{color.OK_GREEN}{'Attack Ended':>23}{color.RESET} "
                          f"{duration}s stable")
                    below_consecutive = 0

            # tidy the console lines
            for _ in range(6):
                sys.stdout.write('\x1b[1A'); sys.stdout.write('\x1b[2K')

        except KeyboardInterrupt:
            design.Output().clear()
            print(f"\r{FORMAT_OUTPUT} Exception: KeyboardInterrupt")
            break
        except Exception as error:
            LOGGER.error("Main loop error: %s", error)
            print(f"{FORMAT_OUTPUT}{'Error:':>23} [{error}]")
            time.sleep(2)

if __name__ == '__main__':
    try:
        design.Output().clear()
        create_directories()
        init()
    except KeyboardInterrupt:
        design.Output().clear()
        print(f"\r{FORMAT_OUTPUT} Exception: KeyboardInterrupt")
        sys.exit(0)
    except Exception as e:
        print(f"{FORMAT_OUTPUT} Fatal error: {e}")
        sys.exit(1)
